/*
  LCD3Wire.cpp - LCD HITACHI 44780 Compatible, 3 Wire library
  Copyright (c) 2008 Alberto J. Medrano Villalobos.  All right reserved.

  Sources:
    - neillzero http://abstractplain.net (based on)
    - The original "LiquidCrystal" 8-bit library and tutorial
      http://www.arduino.cc/en/uploads/Tutorial/LiquidCrystal.zip
      http://www.arduino.cc/en/Tutorial/LCDLibrary
    - DEM 16216 datasheet http://www.maplin.co.uk/Media/PDFs/N27AZ.pdf
    - Massimo's suggested 4-bit code (I took initialization from here) http://www.arduino.cc/cgi-bin/yabb2/YaBB.pl?num=1144924220/8
  See also:
    - glasspusher's code (probably more correct): http://www.arduino.cc/cgi-bin/yabb2/YaBB.pl?num=1160586800/0#0

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  Mod/Written in Venezuela, Fundación CENDITEL
*/

#include "LCD3Wire.h"
extern "C" {
//  #include <stdio.h>       // not needed yet
  #include <string.h>      // needed for strlen()
  #include <inttypes.h>
  #include "WConstants.h"  // all things wiring / arduino
}

//command bytes for LCD
#define CMD_CLR 0x01
#define CMD_RIGHT 0x1C
#define CMD_LEFT 0x18
#define CMD_HOME 0x02

// --------- PINS -------------------------------------
//RS, RW and Enable can be set to whatever you like
int RS_DT = 11;
int Enable = 10;
int CLK = 12;

//--------------------------------------------------------

//how many lines has the LCD? (don't change here - specify on calling constructor)
int g_num_lines = 1;

//pulse the Enable pin high (for a microsecond).
//This clocks whatever command or data is in DB4~7 into the LCD controller.
void LCD3Wire::pulseEnablePin(){
  digitalWrite(Enable,LOW);
  delayMicroseconds(1);
  // send a pulse to enable
  digitalWrite(Enable,HIGH);
  delayMicroseconds(1);
  digitalWrite(Enable,LOW);
  delay(1);  // pause 1 ms.  TODO: what delay, if any, is necessary here?

}

// push data through the 74LS164 (SIPO Register) to LCD's DB0~7 pins, in eight step,
// clocking each with the Enable pin.
void LCD3Wire::pushByte(int value){
  for (int i = 0; i < 8; i++) {
   digitalWrite(RS_DT, (value & 0x80) ? HIGH : LOW);
   clockToSIPO();
   value <<= 1;
  } 
}

void LCD3Wire::clockToSIPO() {
  digitalWrite(CLK,LOW);
  delayMicroseconds(1);
  digitalWrite(CLK,HIGH);
  delayMicroseconds(1);
//  delay(100); //TEMPORAL
  digitalWrite(CLK,LOW);
  delayMicroseconds(1);
//  delay(100); //TEMPORAL
}

//stuff the library user might call---------------------------------
//constructor.  num_lines must be 1 or 2, currently.
LCD3Wire::LCD3Wire (int num_lines) {
  g_num_lines = num_lines;
  if (g_num_lines < 1 || g_num_lines > 2)
  {int
    g_num_lines = 1;
  }
}


void LCD3Wire::commandWrite(int value) {
  pushByte(value);
  //TODO: perhaps better to add a delay after EVERY command, here.  many need a delay, apparently.
  digitalWrite(RS_DT, LOW);
  pulseEnablePin();
}


//print the given character at the current cursor position. overwrites, doesn't insert.
void LCD3Wire::print(int value) {
  //let pushByte worry about the intricacies of Enable, nibble order.
  pushByte(value);
  //set the RS and RW pins to show we're writing data
  digitalWrite(RS_DT, HIGH);
  pulseEnablePin();
  digitalWrite(RS_DT, LOW);
}


//print the given string to the LCD at the current cursor position.  overwrites, doesn't insert.
//While I don't understand why this was named printIn (PRINT IN?) in the original LiquidCrystal library, I've preserved it here to maintain the interchangeability of the two libraries.
void LCD3Wire::printIn(char msg[]) {
  uint8_t i;  //fancy int.  avoids compiler warning when comparing i with strlen()'s uint8_t
  for (i=0;i < strlen(msg);i++){
    print(msg[i]);
  }
}


//send the clear screen command to the LCD
void LCD3Wire::clear(){
  commandWrite(CMD_CLR);
  delay(1);
}


// initiatize lcd after a short pause
//while there are hard-coded details here of lines, cursor and blink settings, you can override these original settings after calling .init()
void LCD3Wire::init () {
  pinMode(Enable,OUTPUT);
  pinMode(RS_DT,OUTPUT);
  pinMode(CLK,OUTPUT);

  delay(50);

  // Sets to 8-bit operation and
  // selects 2-line display and 5 × 8
  commandWrite(0x38);
  delayMicroseconds(60);

  // Turns on display and cursor off.
  // Entire display is in space mode
  // because of initialization
  commandWrite(0x0C);
  delayMicroseconds(60);

  // Sets mode to increment the
  // address by one and to shift the
  // cursor to the right at the time of
  // write to the DD/CGRAM.
  // Display is not shifted.
  commandWrite(0x06);
  delayMicroseconds(60);

  // Clears entire display and sets
  // DDRAM address 0 in address
  // counter.
  commandWrite(0x01);
  delay(1);
}


//non-core stuff --------------------------------------
//move the cursor to the given absolute position.  line numbers start at 1.
//if this is not a 2-line LCD3Wire instance, will always position on first line.
void LCD3Wire::cursorTo(int line_num, int x){
  //first, put cursor home
  commandWrite(CMD_HOME);

  //if we are on a 1-line display, set line_num to 1st line, regardless of given
  if (g_num_lines==1){
    line_num = 1;
  }
  //offset 40 chars in if second line requested
  if (line_num == 2){
    x += 40;
  }
  //advance the cursor to the right according to position. (second line starts at position 40).
  for (int i=0; i<x; i++) {
    commandWrite(0x14);
  }
}

//scroll whole display to left
void LCD3Wire::leftScroll(int num_chars, int delay_time){
  for (int i=0; i<num_chars; i++) {
    commandWrite(CMD_LEFT);
    delay(delay_time);
  }
}

void LCD3Wire::SpanishCharSet(){
  commandWrite(0x40);   // CGRAM Location 0
//  delay(10);
  print(0x00000010);    // ···#· -> 00h ó CGRAM(1)
  print(0x00000100);    // ··#··
  print(0x00001110);    // ·###·
  print(0x00000001);    // ····#
  print(0x00001111);    // ·####
  print(0x00010001);    // #···#
  print(0x00001111);    // ·####
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 01h ó CGRAM(2)
  print(0x00000100);    // ·····
  print(0x00001110);    // ·····
  print(0x00010001);    // ·····
  print(0x00011111);    // ·····
  print(0x00010000);    // ·····
  print(0x00001110);    // ·····
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 02h ó CGRAM(3)
  print(0x00000100);    // ·····
  print(0x00000000);    // ·····
  print(0x00001100);    // ·····
  print(0x00000100);    // ·····
  print(0x00000100);    // ·····
  print(0x00001110);    // ·····
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 03h ó CGRAM(4)
  print(0x00000100);    // ·····
  print(0x00000000);    // ·····
  print(0x00001110);    // ·····
  print(0x00010001);    // ·····
  print(0x00010001);    // ·····
  print(0x00001110);    // ·····
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 04h ó CGRAM(5)
  print(0x00000100);    // ·····
  print(0x00000000);    // ·····
  print(0x00010001);    // ·····
  print(0x00010001);    // ·····
  print(0x00010011);    // ·····
  print(0x00001101);    // ·····
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 05h ó CGRAM(6)
  print(0x00000100);    // ·····
  print(0x00000000);    // ·····
  print(0x00010110);    // ·····
  print(0x00011001);    // ·····
  print(0x00010001);    // ·····
  print(0x00010001);    // ·····
  print(0x00000000);    // ·····

  print(0x00000010);    // ····· -> 06h ó CGRAM(7)
  print(0x00000100);    // ·····
  print(0x00010001);    // ·····
  print(0x00011001);    // ·····
  print(0x00010101);    // ·····
  print(0x00010011);    // ·····
  print(0x00010001);    // ·····
  print(0x00000000);    // ·····

  print(0x00000000);    // ····· -> 07h ó CGRAM(8)
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····
  print(0x00000000);    // ·····

//  delay(10);
  commandWrite(0x80);
//  delay(10);

  }


