/*  keypad v0.3

    Copyright (2007) Mark Stanley

    Author: Mark Stanley <mstanley@technologist.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the 

    Free Software Foundation, Inc.
    59 Temple Place, Suite 330
    Boston, MA  02111-1307
    USA

  *****************************************************************************

    Keypad.cpp is an Arduino library for reading a matrix style keypad. You can 
    use it for any size keypad up to a 4x4 matrix. You could also extend it to 
    more than 4x4 but that has been the largest keypad I've ever seen so that is 
    where I stopped. 

    You can create your keypad with "keypad kpd=keypad()" which will default to a 
    keypad with 4 rows by 3 columns or you can call it with 
    "keypad kpd=keypad(int rows, int cols)" where rows and cols are the number of 
    rows and columns you want for your keypad.

    Look under PIN MAPPINGS to see what you need to do to wire up your keypad. Or 
    you could wire up your keypad and change the col and row mappings to 
    match your wiring.

    Return value: get_key() returns a char from the keymap when a key is pressed.
                            Returns a null ('\0') when no key is pressed.

    Usage:  see the example folder of this library distribution.
*/

#include "keypad.h"

extern "C" {
	#include "WConstants.h"  //all things wiring / arduino
}

// --------- PIN MAPPINGS -------------------------------
// The pin mappings are matched to the pin numbers found on the Arduino. For example
// the first column from your keypad is connected to pin 6 on the Arduino and the
// second column is connected to pin 5. It's the same for the rows.
// These are the numbers you would change to match your keypad wiring. Even though
// they are shown in order you can set them to any valid (random) pin number.

#define COL0	7
#define COL1	8
#define COL2	9
#define COL3	9
#define ROW0	3
#define ROW1	4
#define ROW2	5
#define ROW3	6

// Note: The column pins will be set as outputs and the row pins will be set as inputs.

// Default row and col pin counts. Don't change these here. You can set them in your
// code when instantiating a new keypad object. See the example project.
int num_rows = 4;
int num_cols = 3;

static int col_select;	// Keeps track of which column to activate/de-activate

// 4x4 keypad key mapping. You shouldn't need to touch this unless you have
// a keypad with different (or more) keys than those listed.
const char keymap[4][5] = {	"123A",
				"456B",
				"789C",
				"*0#D"  };

int row[4] = {ROW0, ROW1, ROW2, ROW3};
int col[4] = {COL0, COL1, COL2, COL3};
//--------------------------------------------------------

// get_key() takes control of the data lines on each call just in case
// another part of the user's code needs to share control. This allows
// them to be multiplexed with, for example, the data lines of an LCD.
char keypad::get_key()
{
	static int temp_key;
	int key = '\0';

	digitalWrite(col[col_select],LOW);			// Activate the current column.

	for(int r=0;r<num_rows;r++)				// Scan all the rows for a key press.
	{
                delay(1);
		pinMode(row[r],INPUT);				// Take control of data lines
		if(digitalRead(row[r]) == LOW)			// Is a key STILL being pressed?
		{
                        delay(1);
			temp_key = keymap[r][col_select];	// Remember which key was pressed.
			return '\0';				// Prevents repeating when a key is held down.
		}
	}

	if( (key == '\0') && (temp_key != '\0') )		// Key was pressed and then released
	{
		key = temp_key;
		temp_key = '\0';				// Get ready for next keypress
		return key;
	}

	digitalWrite(col[col_select],HIGH);			// De-activate the current column.

	col_select++;						// Move to the next column.
	if(col_select == num_cols)
		col_select=0;

	return key;
}

// Default - Setup up the keypad in accordance with keymap above.
void keypad::init ()
{
	for (int c=0; c<num_cols; c++)
	{
		pinMode(col[c],OUTPUT);				// Set column pins as outputs for writing
		digitalWrite(col[c],HIGH);			// Make all columns inactive to start with.
	}
	col_select = 0;						// Start with the first column.
}

keypad::keypad()
{
	// Default constructor
}

// Constructor for setting your keypad matrix size
keypad::keypad(int r, int c)
{
	num_rows = r;
	num_cols = c;
}
